const postService = require("../../services/post.service");
const seoService = require("../../services/seo.service");
const { validatePost } = require("../../validators/post.validator");
const { success, error } = require("../../utils/response");

function toBool(value) {
  if (typeof value === "boolean") return value;
  if (typeof value === "string") {
    return value.toLowerCase() === "true" || value === "1";
  }
  if (typeof value === "number") {
    return value === 1;
  }
  return false;
}


exports.createPost = async (req, res, next) => {
  try {
    const body = req.body;

    if (typeof body.tags === "string") {
      body.tags = JSON.parse(body.tags);
    }
    if (typeof body.seo === "string") {
      body.seo = JSON.parse(body.seo);
    }

    const { valid, errors } = validatePost(body);
    if (!valid) return error(res, errors.join(", "), 400);

    // 🔥 map flags
    body.isHero        = toBool(body.isHero);
    body.isFeatured    = toBool(body.isFeatured);
    body.isMustRead    = toBool(body.isMustRead);
    body.isTrending    = toBool(body.isTrending);
    body.isTopThisWeek = toBool(body.isTopThisWeek);

    const post = await postService.createPost(body);
    await seoService.createSeoForPost(post.id, body.seo);

    const fullPost = await postService.getPostBySlug(post.slug);

    return success(res, fullPost, "Post created", 201);
  } catch (err) {
    next(err);
  }
};


exports.updatePost = async (req, res, next) => {
  try {
    const { id } = req.params;
    const body = req.body;

    if (typeof body.tags === "string") {
      body.tags = JSON.parse(body.tags);
    }
    if (typeof body.seo === "string") {
      body.seo = JSON.parse(body.seo);
    }

    const { valid, errors } = validatePost(body);
    if (!valid) return error(res, errors.join(", "), 400);

    body.isHero        = toBool(body.isHero);
    body.isFeatured    = toBool(body.isFeatured);
    body.isMustRead    = toBool(body.isMustRead);
    body.isTrending    = toBool(body.isTrending);
    body.isTopThisWeek = toBool(body.isTopThisWeek);

    await postService.updatePost(id, body);
    await seoService.updateSeoForPost(Number(id), body.seo);

    const fullPost = await postService.getPostBySlug(body.slug);

    return success(res, fullPost, "Post updated");
  } catch (err) {
    next(err);
  }
};


exports.deletePost = async (req, res, next) => {
  try {
    const { id } = req.params;

    await postService.deletePost(id);
    return success(res, null, "Post deleted");
  } catch (err) {
    next(err);
  }
};
