// src/controllers/admin/upload.controller.js
const fs = require("fs");
const path = require("path");
const multer = require("multer");
const env = require("../../config/env");
const { success, error } = require("../../utils/response");

// Make sure we have a valid string path
const uploadDir = env.UPLOAD_DIR; // already absolute from env.js

if (!uploadDir || typeof uploadDir !== "string") {
  throw new Error(`UPLOAD_DIR is invalid: ${uploadDir}`);
}

// Ensure upload directory exists
if (!fs.existsSync(uploadDir)) {
  fs.mkdirSync(uploadDir, { recursive: true });
}

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const ext = path.extname(file.originalname);
    const base = path
      .basename(file.originalname, ext)
      .replace(/\s+/g, "-")
      .toLowerCase();
    const filename = `${base}-${Date.now()}${ext}`;
    cb(null, filename);
  }
});

const upload = multer({ storage });

const handleUpload = (req, res) => {
  if (!req.file) return error(res, "No file uploaded", 400);

  // Public URL (we will mount /uploads in app.js)
  const urlPath = `/uploads/${req.file.filename}`;

  return success(res, { url: urlPath }, "File uploaded");
};

module.exports = {
  upload,
  handleUpload
};
